/**
 * i-net software provides programming examples for illustration only, without warranty
 * either expressed or implied, including, but not limited to, the implied warranties
 * of merchantability and/or fitness for a particular purpose. This programming example
 * assumes that you are familiar with the programming language being demonstrated and
 * the tools used to create and debug procedures. i-net software support professionals
 * can help explain the functionality of a particular procedure, but they will not modify
 * these examples to provide added functionality or construct procedures to meet your
 * specific needs.
 *
 * Copyright © 1999-2025 i-net software GmbH, Berlin, Germany.
**/
package sign;

import java.io.FileOutputStream;
import com.inet.report.Area;
import com.inet.report.Engine;
import com.inet.report.ReportException;
import com.inet.report.Section;
import com.inet.report.SignatureForm;
import com.inet.report.SignatureProperties;

/**
 * This sample shows how to create a visible signature form with a default certificate info factory. The default
 * certificate info factory is defined through configuration settings: Export/PDF. The visible signature can display two
 * elements each of them lays in its own section: Subject name and Certificate Details. Text of these elements will be
 * displayed with the SansSerif font. Some options control the signature view: Both elements must be shown or only one
 * of them. In the latter case the position of the section, containing Subject name or empty section if Subject name is
 * not shown (right or left). Certificate details are consist of some items that can be independently chosen. For using
 * this sample some settings must be adjusted: they are listed below as 'Parameters to adjust'.
 */
public class VisibleSignatureSample {
    // Parameters to adjust:
    private String reportName = "E:/sign/reports/EmptyReport.rpt";
    private String exportPath = "E:/sign/export/testVS.pdf";

    /**
     * main method
     * @param args arguments as Strings : not used.
     */
    public static void main( String[] args ) {
        new VisibleSignatureSample().writeVisibleSignature();
    }

    /**
     * Creates engine an empty engine; adds signature form to section elements; executes pdf export.
     */
    private void writeVisibleSignature() {
        try {
            Engine eng = createSampleReport( reportName );
            addSignatureFormToReport( eng );
            doPdfExport( eng, exportPath );
        } catch( Exception e ) {
            e.printStackTrace();
        }
    }

    /**
     * Creates engine empty engine with some prompt fields describing security settings.
     * @param reportName absolute path (e.g. "C:/reports/empty.rpt")
     * @return test engine
     * @throws ReportException if creating engine or setting report file fails.
     */
    private Engine createSampleReport( String reportName ) throws ReportException {
        Engine eng = new Engine( Engine.EXPORT_PDF );
        eng.setReportFile( reportName );
        return eng;
    }

    /**
     * Defines the items that must be shown in the signature field.
     * @param eng Engine.
     * @throws ReportException if adding some elements to the signature form failed.
     */
    private void addSignatureFormToReport( Engine eng ) throws ReportException {
        Area area = eng.getArea( "RH" );
        Section sec = area.getSection( 0 );

        // SignatureForm position and size: all values in twips:
        int x = 40;
        int y = 40;
        int width = 2400;
        int height = 1600;

        SignatureForm sigForm = sec.addSignatureForm( x, y, width, height );
        //subject name will be shown in its own section:
        sigForm.setShowSubjectName( true );

        // the section with subject name will be in the right: 
        sigForm.setTextPosition( SignatureProperties.POSITION_RIGHT );

        // certificate options:
        // show labels 
        sigForm.setWriteCertificateLabels( true );

        // in this case the subject name will be shown in certificate details:
        sigForm.setWriteCertificateSubjectName( true );

        // distinguished name will be shown  in certificate details:
        sigForm.setWriteCertificateDistinguishedName( true );

        // date (computer time) will be shown  in certificate details:
        sigForm.setWriteCertificateDate( true );

        // the field will be divided in two sections(subject name and certificate details)
        // sigForm.setDivideField(true);

    }

    /**
     * Runs pdf export and saves the result in a pdf file.
     * @param eng active engine.
     * @param exportPath path of pdf file.
     */
    private void doPdfExport( Engine eng, String exportPath ) {
        FileOutputStream fos = null;
        try {
            eng.execute();
            fos = new FileOutputStream( exportPath );
            for( int i = 1; i <= eng.getPageCount(); i++ ) {
                fos.write( eng.getPageData( i ) );
            }
        } catch( Throwable th ) {
            th.printStackTrace();
        } finally {
            if( fos != null ) {
                try {
                    fos.close();
                } catch( Exception e ) {/*empty*/
                }
            }
        }
    }

}
